<?php

namespace App\Http\Controllers\Admin;

use App\DTOs\Job\JobCreateDTO;
use App\Http\Controllers\Controller;
use App\Http\Requests\Admin\JobCreateRequest;
use App\Http\Requests\Admin\JobUpdateRequest;
use App\Services\JobService;
use Illuminate\Http\RedirectResponse;
use Illuminate\View\View;

class JobController extends Controller
{
    public function __construct(protected JobService $jobService)
    {
    }

    public function index(): View
    {
        $listData = $this->jobService->getPaginated();
        return view('admin.jobs.index', compact('listData'));
    }

    public function create(): View
    {
        return view('admin.jobs.create');
    }

    public function edit($id): View
    {
        $job = $this->jobService->find($id);
        if (!$job) {
            abort(404);
        }
        return view('admin.jobs.edit', compact('job'));
    }

    public function store(JobCreateRequest $request): RedirectResponse
    {
        $dto = new JobCreateDTO($request->validated());
        $job = $this->jobService->create($dto);

        if (!$job) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'Ocurrió un error al guardar el cargo. Intenta nuevamente.']);
        }

        return redirect()->route('admin.jobs.index')->with('success', 'Cargo creado correctamente.');
    }

    public function update(JobUpdateRequest $request, $id): RedirectResponse
    {
        $dto = new JobCreateDTO($request->validated());
        $job = $this->jobService->update($id, $dto);

        if (!$job) {
            return redirect()->back()
                ->withInput()
                ->withErrors(['error' => 'No se pudo actualizar el cargo.']);
        }

        return redirect()->route('admin.jobs.index')->with('success', 'Cargo actualizado correctamente.');
    }

    public function destroy($id): RedirectResponse
    {
        $deleted = $this->jobService->delete($id);
        if ($deleted) {
            return redirect()->route('admin.jobs.index')->with('success', 'Cargo eliminado correctamente.');
        }
        return redirect()->route('admin.jobs.index')->with('error', 'No se pudo eliminar el cargo.');
    }
}
