<?php

namespace App\Http\Requests\Lead;

use Illuminate\Foundation\Http\FormRequest;
use Illuminate\Validation\Rule;

class UpdateLeadRequest extends FormRequest
{
    public function authorize(): bool
    {
        return true;
    }

    public function rules(): array
    {
        $user = auth()->user();
        return [
            'users_assigned_ids' => [
                Rule::requiredIf(fn() => $user && $user->hasAnyRole(['admin', 'superadmin'])),
                'array',
                'min:1'
            ],
            'status_pipeline_id' => ['required', 'integer', 'exists:status_pipelines,id'],
            'lead_type_id' => ['required', 'integer', 'exists:lead_types,id'],
            'name_event' => ['required', 'string', 'max:255'],
            'venue_id' => ['required', 'integer', 'exists:venues,id'],
            'ability_max' => ['required', 'integer', 'min:1'],
            'number_of_people' => ['required', 'integer', 'min:1'],
            'rooms_guests' => ['required', 'integer', 'min:1'],
            'estimated_budget' => ['required', 'numeric', 'min:0'],
            'total_cost' => ['nullable', 'numeric', 'min:0'],
            'money_budget' => ['required', 'string'],

            'name' => ['required', 'string', 'max:100'],
            'last_name' => ['required', 'string', 'max:100'],
            'email' => ['required', 'string', 'email', 'max:255'],
            'phone' => ['nullable', 'string', 'max:20'],
            'additional_information' => ['nullable', 'string'],
            'country' => ['nullable', 'string', 'max:255'],

            'is_flexible' => ['required', 'boolean'],
            'flexible_month' => ['required_if:is_flexible,true', 'nullable', 'string'],
            'flexible_duration' => ['required_if:is_flexible,true', 'nullable', 'integer', 'min:1'],

            'checkin_at' => ['required_if:is_flexible,false', 'nullable', 'date'],
            'checkout_at' => ['required_if:is_flexible,false', 'nullable', 'date', 'after:checkin_at'],
        ];
    }

    public function messages(): array
    {
        return [
            'users_assigned_ids.required' => 'Los miembros del personal son requeridos.',
            'users_assigned_ids.min' => 'Debes seleccionar al menos un miembro del listado.',
            'users_assigned_ids.array' => 'El formato de asignación no es válido.',
            'users_assigned_ids.*.exists' => 'Uno de los usuarios seleccionados no es válido.',

            'status_pipeline_id.required' => 'El estado del evento es obligatorio.',
            'status_pipeline_id.exists' => 'El estado del evento seleccionado no es válido.',

            'lead_type_id.required' => 'El tipo de evento es obligatorio.',
            'lead_type_id.exists' => 'El tipo de evento seleccionado no es válido.',

            'name_event.required' => 'El nombre del evento es obligatorio.',
            'name_event.max' => 'El nombre del evento no puede superar los :max caracteres.',

            'venue_id.required' => 'Debes seleccionar un Venue.',
            'venue_id.exists' => 'El Venue seleccionado no es válido.',

            'ability_max.required' => 'La capacidad máxima es obligatoria.',
            'ability_max.min' => 'La capacidad máxima debe ser al menos :min.',

            'number_of_people.required' => 'El número de asistentes es obligatorio.',
            'number_of_people.min' => 'El número de asistentes debe ser al menos :min.',
            'number_of_people.lte' => 'El número de asistentes no puede exceder la capacidad máxima.',

            'rooms_guests.required' => 'El número de habitaciones es obligatorio.',
            'rooms_guests.min' => 'El número de habitaciones debe ser al menos :min.',

            'estimated_budget.required' => 'El presupuesto estimado es obligatorio.',
            'estimated_budget.numeric' => 'El presupuesto debe ser un valor numérico.',
            'estimated_budget.min' => 'El presupuesto debe ser un valor positivo o cero.',

            'money_budget.required' => 'La moneda del presupuesto es obligatoria.',

            'name.required' => 'El nombre del contacto es obligatorio.',
            'name.max' => 'El nombre no puede superar los :max caracteres.',

            'last_name.required' => 'El apellido del contacto es obligatorio.',
            'last_name.max' => 'El apellido no puede superar los :max caracteres.',

            'email.required' => 'El correo electrónico es obligatorio.',
            'email.email' => 'El formato del correo electrónico no es válido.',
            'email.max' => 'El correo electrónico no puede superar los :max caracteres.',
            'country.max' => 'El país no puede superar los :max caracteres.',
            'phone.max' => 'El número de teléfono no puede superar los :max caracteres.',
            'additional_information.string' => 'La información adicional debe ser texto.',

            'is_flexible.boolean' => 'El campo de fechas flexibles no es válido.',

            'checkin_at.required_if' => 'La fecha de llegada es obligatoria.',
            'checkin_at.date' => 'La fecha de llegada debe ser una fecha válida.',

            'checkout_at.required_if' => 'La fecha de salida es obligatoria.',
            'checkout_at.date' => 'La fecha de salida debe ser una fecha válida.',
            'checkout_at.after' => 'La fecha de salida debe ser posterior a la fecha de llegada.',

            'flexible_month.required_if' => 'El mes es obligatorio.',
            'flexible_month.string' => 'El mes flexible debe ser un texto válido.',

            'flexible_duration.required_if' => 'Número de noches es obligatorio.',
            'flexible_duration.integer' => 'La duración flexible debe ser un número entero.',
            'flexible_duration.min' => 'La duración flexible debe ser de al menos :min día.',

            'total_cost.numeric' => 'El costo total debe ser un valor numérico.',
            'total_cost.min' => 'El costo total no puede ser un valor negativo.',
        ];
    }

    protected function prepareForValidation()
    {
        $this->merge([
            'is_flexible' => filter_var($this->is_flexible, FILTER_VALIDATE_BOOLEAN),
        ]);
    }
}
