<?php

namespace App\Models;

// use Illuminate\Contracts\Auth\MustVerifyEmail;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Notifications\Notifiable;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Database\Eloquent\Casts\Attribute;

class User extends Authenticatable
{
    use HasApiTokens, HasFactory, Notifiable;

    /**
     * The attributes that are mass assignable.
     *
     * @var array<int, string>
     */
    protected $fillable = [
        'name',
        'email',
        'password',
        'active'
    ];

    /**
     * The attributes that should be hidden for serialization.
     *
     * @var array<int, string>
     */
    protected $hidden = [
        'password',
        'remember_token',
    ];

    /**
     * The attributes that should be cast.
     *
     * @var array<string, string>
     */
    protected $casts = [
        'email_verified_at' => 'datetime',
        'password' => 'hashed',
    ];

    public function staftDetail()
    {
        return $this->hasOne(StaftDetail::class, 'user_id');
    }

    public function roles()
    {
        return $this->belongsToMany(Role::class);
    }

    public function hasRole($roleName)
    {
        return $this->roles->contains('name', $roleName);
    }

    public function hasAnyRole(array $roles): bool
    {
        return $this->roles->pluck('name')->intersect($roles)->isNotEmpty();
    }

    public function leadTypes(): BelongsToMany
    {
        return $this->belongsToMany(LeadType::class, 'lead_type_user', 'user_id', 'lead_type_id');
    }

    protected function fullName(): Attribute
    {
        return Attribute::make(
            get: function () {
                $firstName = $this->attributes['name'] ?? '';
                $lastName = $this->staftDetail->last_name ?? '';
                return trim("{$firstName} {$lastName}");
            }
        )->shouldCache();
    }

    protected function profilePhotoUrl(): Attribute
    {
        return Attribute::make(
            get: function () {
                $document = $this->staftDetail?->documents
                    ->first();
                if ($document) {
                    return asset('storage/' . $document->file_path);
                }
                return 'https://ui-avatars.com/api/?name=' . urlencode($this->name);
            }
        );
    }
    public function assignedLeads(): BelongsToMany
    {
        return $this->belongsToMany(Lead::class, 'lead_user')
            ->withPivot('assignment_method')
            ->withTimestamps();
    }
}