<?php

namespace App\Services;

use App\DTOs\Lead\LeadApiCreateDTO;
use App\DTOs\Lead\LeadCreateDTO;
use App\DTOs\Lead\LeadUpdateDTO;
use App\Http\Resources\Lead\LeadListResource;
use App\Models\Lead;
use App\Models\User;
use App\Queries\LeadListByUser;
use App\Queries\LeadListQuery;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Collection as BaseCollection;
use Spatie\Activitylog\Models\Activity;

class LeadService
{
    public function get(): Collection
    {
        return Lead::orderBy('name', 'asc')->get();
    }

    public function getPaginated(User $user, array $filters = [], int $perPage = 10): LengthAwarePaginator
    {
        $paginator = LeadListQuery::build($user, $filters)->paginate($perPage);

        return $paginator->through(function ($lead) {
            return (new LeadListResource($lead))->resolve(); 
        });
    }

    public function getLeadsByUserPaginated(string $user_id, int $perPage = 10): LengthAwarePaginator
    {
        $paginator = LeadListByUser::build($user_id)->paginate($perPage);

        return $paginator->through(function ($lead) {
            $assignment = $lead->assignedUsers->first();
            return [
                'name_event' => $lead->name_event,
                'name_lead_type' => $lead->leadType?->name,
                'status_pipeline' => $lead->statusPipeline,
                'assigned_at' => $assignment ? $assignment->pivot->created_at->toDateTimeString() : null,
            ];
        });
    }

    public function find(string $slug): ?Lead
    {
        try {
            return Lead::with([
                'assignedUsers.staftDetail.documents',
                'assignedUsers.roles'
            ])
            ->where('slug', $slug)
            ->first();
        } catch (Exception $e) {
            Log::error('Error al encontrar el lead: ' . $e->getMessage(), [
                'data' => $slug,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function getTimeline(Lead $lead, int $perPage = 10, ?int $lastActivityId = null, string $order = 'desc'): BaseCollection
    {
        $activitiesQuery = $lead->activities();
        if ($order === 'asc') {
            $activitiesQuery->oldest();
            if ($lastActivityId) {
                $activitiesQuery->where('id', '>', $lastActivityId);
            }
        } else {
            $activitiesQuery->latest();
            if ($lastActivityId) {
                $activitiesQuery->where('id', '<', $lastActivityId);
            }
        }

        $activities = $activitiesQuery
            ->take($perPage)
            ->with([
                'causer' => function ($query) {
                    $query->with('staftDetail.documents');
                }
            ])
            ->get()
            ->map(function (Activity $activity) {
                $causer = $activity->causer;
                return [
                    'type' => 'activity',
                    'created_at' => $activity->created_at,
                    'created_since' => $activity->created_at->diffForHumans(),
                    'actor_name' => $causer->fullName ?? $causer->name ?? 'Landing',
                    'actor_id' => $activity->causer_id,
                    'photo_url' => $causer->profile_photo_url ?? null,
                    'description' => $activity->description,
                    'log_name' => $activity->log_name,
                    'properties' => $activity->properties->toArray(),
                    'id' => $activity->id,
                    'event' => $activity->event,
                ];
            });

        return $activities;
    }

    public function getCommentsWithDocuments(Lead $lead, int $perPage = 10, ?int $lastCommentId = null, string $order = 'desc'): Collection
    {
        try {
            $commentsQuery = $lead->comments();
            $commentsQuery->with(['user', 'documents']);
            if ($lastCommentId) {
                if ($order === 'asc') {
                    $commentsQuery->where('id', '>', $lastCommentId);
                } else {
                    $commentsQuery->where('id', '<', $lastCommentId);
                }
            }
            if ($order === 'asc') {
                $commentsQuery->orderBy('created_at', 'asc');
            } else {
                $commentsQuery->orderBy('created_at', 'desc');
            }
            $comments = $commentsQuery
                ->take($perPage)
                ->get();

            return $comments;
        } catch (Exception $e) {
            Log::error('Error al obtener los comentarios y documentos del lead.', [
                'lead_id' => $lead->id,
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);
            return collect();
        }
    }

    public function create(LeadCreateDTO $dto, array $usersAssignment = []): ?Lead
    {
        try {
            $lead = Lead::create($dto->toArray());
            if ($lead && !empty($usersAssignment)) {
                $lead->assignedUsers()->sync($usersAssignment);
            }
            return $lead;
        } catch (Exception $e) {
            Log::error('Error al crear el lead: ' . $e->getMessage(), [
                'data' => $dto,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function createWithApi(LeadApiCreateDTO $dto): ?Lead
    {
        try {
            $firstStatus = \App\Models\StatusPipeline::orderBy('order_position', 'asc')->first();
            if (!$firstStatus) {
                Log::error('API Lead Error Create: No se encontró ningún StatusPipeline.');
                return null;
            }
            $dto->status_pipeline_id = $firstStatus->id;
            return Lead::create($dto->toArray());
        } catch (Exception $e) {
            Log::error('Error al crear el lead desde la api: ' . $e->getMessage(), [
                'data' => $dto,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function update(int $id, LeadUpdateDTO $dto, array $syncData = []): ?Lead
    {
        try {
            $lead = Lead::find($id);
            if ($lead) {
                $lead->update($dto->toArray());
                if (!empty($syncData)) {
                    $lead->assignedUsers()->sync($syncData);
                }
            }
            return $lead;
        } catch (Exception $e) {
            Log::error('Error al actualizar el lead: ' . $e->getMessage(), [
                'data' => $dto,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }
}