<?php

namespace App\Services;

use App\DTOs\LeadType\LeadTypeCreateDTO;
use App\Models\LeadType;
use Exception;
use Illuminate\Support\Facades\Log;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Collection;

class LeadTypeService
{
    public function get():Collection
    {
        return LeadType::where('active', true)->orderBy('name', 'desc')->get();
    }

    public function getPaginated(int $perPage = 10): LengthAwarePaginator
    {
        return LeadType::orderBy('name', 'asc')->paginate($perPage);
    }

    public function find(int $id): ?LeadType
    {
        try {
            return LeadType::find($id);
        } catch (Exception $e) {
            Log::error('Error al encontrar tipo de evento: ' . $e->getMessage(), [
                'data' => $id,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function create(LeadTypeCreateDTO $leadTypeDTO): ?LeadType
    {
        try {
            return LeadType::create($leadTypeDTO->toArray());
        } catch (Exception $e) {
            Log::error('Error al crear el tipo de evento: ' . $e->getMessage(), [
                'data' => $leadTypeDTO,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function update(int $id, LeadTypeCreateDTO $leadTypeDTO): ?LeadType
    {
        try {
            $leadType = LeadType::find($id);
            if ($leadType) {
                $leadType->update($leadTypeDTO->toArray());
            }
            return $leadType;
        } catch (Exception $e) {
            Log::error('Error al actualizar el tipo de evento: ' . $e->getMessage(), [
                'id' => $id,
                'data' => $leadTypeDTO,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function delete(int $id): bool
    {
        try {
            $typeEventDelete = LeadType::find($id);
            return $typeEventDelete ? $typeEventDelete->delete() : false;
        } catch (Exception $e) {
            Log::error('Error al eliminar tipo de evento: ' . $e->getMessage(), [
                'id' => $id,
                'trace' => $e->getTraceAsString(),
            ]);
            return false;
        }
    }
}