<?php

namespace App\Services;

use App\DTOs\Venue\VenueCreateDto;
use App\Models\Venue;
use App\Support\OperationResult;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Collection;
use Illuminate\Support\Facades\Log;

class VenueService
{
    public function get(): Collection
    {
        return Venue::where('active', true)->orderBy('name', 'asc')->get();
    }

    public function getPaginated(int $perPage = 10): LengthAwarePaginator
    {
        return Venue::orderBy('name', 'asc')->paginate($perPage);
    }

    public function find(int $id): ?Venue
    {
        try {
            return Venue::find($id);
        } catch (Exception $e) {
            Log::error('Error al encontrar el lugar: ' . $e->getMessage(), [
                'data' => $id,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function create(VenueCreateDto $venueCreateDto): ?Venue
    {
        try {
            $data = $venueCreateDto->toArray();

            return Venue::create($data);
        } catch (Exception $e) {
            Log::error('Error al crear el venue: ' . $e->getMessage(), [
                'data' => $venueCreateDto->toArray(),
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function update(int $id, VenueCreateDto $venueCreateDto): ?Venue
    {
        try {
            $venueUpdate = Venue::find($id);
            if ($venueUpdate) {
                $venueUpdate->update($venueCreateDto->toArray());
            }
            return $venueUpdate;
        } catch (Exception $e) {
            Log::error('Error al actualizar venue: ' . $e->getMessage(), [
                'id' => $id,
                'data' => $venueCreateDto->toArray(),
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function delete(int $id): OperationResult
    {
        try {
            $deleteVenue = Venue::find($id);
            if (!$deleteVenue) {
                return OperationResult::error("El venue no existe.");
            }
            if ($deleteVenue->leads()->exists()) {
                return OperationResult::error("No se puede eliminar el venue porque tiene leads asociados.");
            }
            $deleteVenue->delete();
            return OperationResult::success(message: "Venue eliminado correctamente.");
        } catch (Exception $e) {
            Log::error('Error al eliminar venue: ' . $e->getMessage(), [
                'id' => $id,
                'trace' => $e->getTraceAsString(),
            ]);
            return OperationResult::error("Hubo un error al eliminar el venue.");
        }
    }
}