<?php

namespace App\Services;

use App\DTOs\Guest\CreateGuestDTO;
use App\DTOs\Guest\UpdateGuestDTO;
use App\Models\Guest;
use Exception;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;

class GuestService
{
    public function getAll(): Collection
    {
        return Guest::all();
    }

    public function getByIds(array $guestIds): Collection
    {
        return Guest::whereIn('id', $guestIds)->get();
    }

    public function getPaginated(int $perPage = 10): LengthAwarePaginator
    {
        return Guest::orderBy('created_at', 'desc')->paginate($perPage);
    }

    public function getNotConfirmedPaginated(int $perPage = 10): LengthAwarePaginator
    {
        return Guest::whereDoesntHave('rsvp')
            ->orderBy('name')
            ->paginate($perPage);
    }

    public function searchByLastName(string $term): Collection
    {
        return Guest::where('name', 'LIKE', '%' . $term . '%')
            ->limit(10)
            ->pluck('name');
    }

    public function getAllGuestNames(): Collection
    {
        return Guest::orderBy('name')->pluck('name');
    }

    public function findBySlug(string $slug): ?Guest{
        return Guest::where('slug', $slug)->first();
    }

    public function existsGuestByName(string $name): ?Guest
    {
        return Guest::where('name', $name)->first();
    }

    public function find(int $id): ?Guest
    {
        return Guest::find($id);
    }

    public function create(CreateGuestDTO $dto): ?Guest
    {
        try {
            return Guest::create([
                "name" => $dto->name,
                "email" => $dto->email,
                "phone" => $dto->phone,
                "max_guests" => $dto->maxGuests,
            ]);
        } catch (Exception $e) {
            Log::error('Error al crear invitado: ' . $e->getMessage(), [
                'data' => $dto,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function update(int $id, UpdateGuestDTO $dto): ?Guest
    {
        try {
            $guest = Guest::find($id);
            if ($guest) {
                $guest->update([
                    "name" => $dto->name,
                    "email" => $dto->email,
                    "phone" => $dto->phone,
                    "max_guests" => $dto->maxGuests,
                ]);
            }
            return $guest;
        } catch (Exception $e) {
            Log::error('Error al actualizar invitado: ' . $e->getMessage(), [
                'id' => $id,
                'data' => $dto,
                'trace' => $e->getTraceAsString(),
            ]);
            return null;
        }
    }

    public function delete(int $id): bool
    {
        $guest = Guest::find($id);
        return $guest ? $guest->delete() : false;
    }
}