<?php
namespace App\Services;

use App\Channels\WhatsAppChannel;
use App\Interfaces\NotificationChannel;
use App\Channels\EmailChannel;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Log;

class NotificationService
{
    protected array $channels;

    public function __construct(EmailChannel $emailChannel, WhatsAppChannel $whatsAppChannel)
    {
        $this->channels = [
            'email' => $emailChannel,
            'whatsapp' => $whatsAppChannel,
        ];
    }

    protected function getChannel(string $key): ?NotificationChannel
    {
        return $this->channels[$key] ?? null;
    }

    public function sendBulkNotification(Collection $guests, string $channelKey): array
    {
        $channel = $this->getChannel($channelKey);

        if (!$channel) {
            Log::warning("Canal de notificación no válido: {$channelKey}");
            return ['error' => 'Canal no válido o no implementado.'];
        }

        $successCount = 0;
        $failCount = 0;
        $details = [];

        foreach ($guests as $guest) {
            try {
                $success = $channel->send($guest);
                if ($success) {
                    $successCount++;
                    $details[] = [
                        'guest' => $guest->name,
                        'email' => $guest->email ?? null,
                        'status' => 'success',
                    ];
                } else {
                    $failCount++;
                    $details[] = [
                        'guest' => $guest->name,
                        'email' => $guest->email ?? null,
                        'status' => 'failed',
                        'error' => 'Error al enviar mensaje.',
                    ];
                }
            } catch (\Throwable $e) {
                $failCount++;
                $details[] = [
                    'guest' => $guest->name,
                    'email' => $guest->email ?? null,
                    'status' => 'failed',
                    'error' => "Error interno, lo sentimos intenta más tarde",
                ];
                Log::error("Error enviando notificación a {$guest->name}: {$e->getMessage()}");
            }
        }

        return [
            $channelKey => [
                'success' => $failCount === 0,
                'total_guests' => $guests->count(),
                'success_count' => $successCount,
                'fail_count' => $failCount,
                'details' => $details,
            ]
        ];
    }
}