const animatedPlane = () => {
  const desktopSvg = document.querySelector('#path-svg');
  const mobileSvg = document.querySelector('#path-svg-mobile');

  const svg = window.innerWidth >= 768 ? desktopSvg : mobileSvg;
  const path = svg.querySelector('path');
  const pathLength = path.getTotalLength();

  const cursor = document.getElementById('cursor-progress');
  const texts = Array.from(document.querySelectorAll('.text-change-animated'));
  const totalTexts = texts.length;

  texts.forEach(text => text.style.opacity = '0');
  texts[0].style.opacity = '1';

  const cursorWidth = 25;
  const cursorHeight = 22;

  function getScaledPoint(length) {
    const point = path.getPointAtLength(length);
    const svgRect = svg.getBoundingClientRect();
    const viewBox = svg.viewBox.baseVal;

    const scaleX = svgRect.width / viewBox.width || 1;
    const scaleY = svgRect.height / viewBox.height || 1;

    return {
      x: point.x * scaleX + svgRect.left,
      y: point.y * scaleY + svgRect.top
    };
  }

  let currentIndex = 0;
  let isTransitioning = false;

  gsap.to({ progress: 0 }, {
    progress: 1,
    scrollTrigger: {
      trigger: '#planeHoursBlock',
      start: 'top top',
      end: '+=100%',
      scrub: true,
      pin: true,
      anticipatePin: 1,
      onUpdate: self => {
        const lengthAtProgress = (1 - self.progress) * pathLength;
        const point = getScaledPoint(lengthAtProgress);

        cursor.style.position = 'fixed';
        cursor.style.left = (point.x - cursorWidth / 2) + 'px';
        cursor.style.top = (point.y - cursorHeight / 2) + 'px';

        const index = Math.floor(self.progress * totalTexts);

        if (index !== currentIndex && !isTransitioning) {
          isTransitioning = true;

          const previous = texts[currentIndex];
          const next = texts[index];

          gsap.to(previous, {
            autoAlpha: 0,
            duration: 0.4,
            ease: 'power2.in',
            onComplete: () => {
              gsap.to(next, {
                autoAlpha: 1,
                duration: 0.5,
                ease: 'power2.out',
                onComplete: () => {
                  currentIndex = index;
                  isTransitioning = false;
                }
              });
            }
          });
        }
      }
    }
  });

  // Nubes y avión
  const cloudScrollOptions = {
    trigger: "#planeHoursBlock",
    start: "top top",
    end: "+=100%",
    scrub: true
  };

  gsap.to(".nube1", { y: 100, x: -50, scrollTrigger: cloudScrollOptions });
  gsap.to(".nubecenter", { y: 80, x: 40, scrollTrigger: cloudScrollOptions });
  gsap.to(".nube3", { y: 60, x: -30, scrollTrigger: cloudScrollOptions });
  gsap.to(".avion-plane", { y: -100, x: 100, scrollTrigger: cloudScrollOptions });

  window.addEventListener('resize', () => ScrollTrigger.refresh());
};

const animatedCard3D = () => {
  const scene = new THREE.Scene();
  const container = document.getElementById('card-animated');

  const width = window.innerWidth;
  const height = window.innerHeight;

  const camera = new THREE.PerspectiveCamera(35, width / height, 0.1, 100);
  camera.position.z = 6.4;

  const renderer = new THREE.WebGLRenderer({ antialias: true, alpha: true });
  renderer.setSize(width, height);
  container.appendChild(renderer.domElement);

  // ✅ Cambiar a absolute
  renderer.domElement.style.position = 'absolute';
  renderer.domElement.style.top = '0';
  renderer.domElement.style.left = '0';
  renderer.domElement.style.width = '100vw';
  renderer.domElement.style.height = '100vh';
  renderer.domElement.style.pointerEvents = 'none'; // evita que bloquee clics

  // Tamaño de la card
  // const cardWidth = 2;
  // const cardHeight = 2.8;
  const cardDepth = 0.5;

  const desiredPixelWidth = 340;
  const desiredPixelHeight = 480;
  const fov = 35;
  const cameraZ = 6.4;
  const canvasHeight = window.innerHeight;
  const fovRad = THREE.MathUtils.degToRad(fov);
  const visibleWorldHeight = 2 * cameraZ * Math.tan(fovRad / 2);
  const unitsPerPixel = visibleWorldHeight / canvasHeight;

  const cardHeight = desiredPixelHeight * unitsPerPixel;
  const cardWidth = desiredPixelWidth * unitsPerPixel;

  const geometry = new RoundedBoxGeometry(cardWidth, cardHeight, cardDepth, 20, 0.5);
  const material = new THREE.MeshStandardMaterial({
    color: 0x2c1a2e,
    metalness: 0.1,
    roughness: 0.7
  });

  const card = new THREE.Mesh(geometry, material);
  card.rotation.z = THREE.MathUtils.degToRad(-18);
  card.position.set(0, 0, 0);
  scene.add(card);

  const light = new THREE.PointLight(0xffffff, 1);
  light.position.set(5, 5, 5);
  scene.add(light);

  function animate() {
    requestAnimationFrame(animate);
    renderer.render(scene, camera);
  }
  animate();

  window.addEventListener('resize', () => {
    const width = window.innerWidth;
    const height = window.innerHeight;
    renderer.setSize(width, height);
    camera.aspect = width / height;
    camera.updateProjectionMatrix();
  });

  ScrollTrigger.create({
    trigger: '#container-scroll-card',
    start: "top top",
    end: "+=744 40%",
    scrub: true,
    pin: true,
    pinSpacing: false,
    markers: false,
    onUpdate: self => {
      // Esto te da un progreso entre 0 y 1
      const progress = self.progress;
      // Actualiza la rotación y cualquier otra propiedad
      card.rotation.y = progress * Math.PI * 2;
      card.rotation.z = (1 - progress) * THREE.MathUtils.degToRad(-18); // para "deshacer" el -18°
      card.position.y = -0.5 * progress;
    }
  });

  gsap.to("#card-w-animated", {
    scaleX: 1,
    opacity: 1,
    ease: "power2.out",
    duration: 1.2,
    scrollTrigger: {
      trigger: "#card-w-animated",
      start: "top 50%",
      end: "+=200",
      toggleActions: "play none none none",
      markers: false,
    }
  });
};

const modalSuccess = () => {
  const modal = document.getElementById('modal-success');
  if (modal) {
    gsap.to(modal, {
      opacity: 1,
      scale: 1,
      duration: 1,
      ease: 'power2.out'
    });
    // Opcional: cerrar al hacer click fuera del modal
    modal.addEventListener('click', (e) => {
      if (e.target === modal) {
        gsap.to(modal, {
          opacity: 0,
          scale: 0.95,
          duration: 0.3,
          ease: 'power2.in',
          onComplete: () => modal.remove()
        });
      }
    });
  }
};