<?php

namespace App\Http\Controllers\Api\Restaurant;

use App\Helpers\FileHelper;
use App\Http\Controllers\Controller;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Exception;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\DB;

class CategoryController extends Controller
{

    public function getAllCategory($idMenu)
    {
        $rows = DB::table('categories as parent')
            ->where('parent.menu_id', $idMenu)
            ->select(
                'parent.category_id as category_id',
                'parent.name as category_name',
                'child.category_id as subcategory_id',
                'child.name as subcategory_name',
            )
            ->leftJoin('categories as child', 'child.parent_id', '=', 'parent.category_id')
            ->whereNull('parent.parent_id')
            ->orderBy('parent.category_id')
            ->get();

        // Agrupar resultados por categoría padre
        $grouped = [];
        foreach ($rows as $row) {
            $catId = $row->category_id;
            if (!isset($grouped[$catId])) {
                $grouped[$catId] = [
                    'id' => $row->category_id,
                    'name' => $row->category_name,
                    'subcategories' => [],
                ];
            }

            if ($row->subcategory_id) {
                $grouped[$catId]['subcategories'][] = [
                    'id' => $row->subcategory_id,
                    'name' => $row->subcategory_name,
                ];
            }
        }

        return response()->json(array_values($grouped), 200);
    }

    public function getCategoryDetail($idCategory)
    {
        $categoryWithSubcategories = DB::table('categories as parent')
            ->select(
                'parent.category_id as category_id',
                'parent.name as category_name',
                'parent.menu_id',
                'parent.image_url',
                'child.category_id as subcategory_id',
                'child.name as subcategory_name'
            )
            ->leftJoin('categories as child', 'child.parent_id', '=', 'parent.category_id')
            ->where('parent.category_id', $idCategory)
            ->get();

        if ($categoryWithSubcategories->isEmpty()) {
            return response()->json(['error' => 'Categoría no encontrada'], 404);
        }

        // Armar la respuesta
        $category = [
            'id' => $categoryWithSubcategories[0]->category_id,
            'name' => $categoryWithSubcategories[0]->category_name,
            'menu_id' => $categoryWithSubcategories[0]->menu_id,
            'image_url' => $categoryWithSubcategories[0]->image_url,
            'subcategories' => []
        ];

        foreach ($categoryWithSubcategories as $row) {
            if ($row->subcategory_id) {
                $category['subcategories'][] = [
                    'id' => $row->subcategory_id,
                    'name' => $row->subcategory_name,
                ];
            }
        }

        return response()->json([$category], 200);
    }

    public function createCategory(Request $request)
    {
        try {
            if ($request->has('subcategories')) {
                $request->merge([
                    'subcategories' => json_decode($request->subcategories, true),
                ]);
            }
            $validatedData = Validator::make($request->all(), [
                'name' => 'required|string|max:500',
                'img_file' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5048',
                'menu_id' => 'required|integer|exists:menus,menu_id',
                'subcategories' => 'nullable|array',
                'subcategories.*.nameSubcategory' => 'required_with:subcategories|string|max:255',
            ]);
            if ($validatedData->fails()) {
                return response()->json(['error' => $validatedData->errors()->toArray()], 400);
            }
            DB::beginTransaction();
            $logoPath = null;
            if ($request->hasFile('img_file')) {
                $logoPath = FileHelper::replace(
                    $request->file('img_file'),
                    '',
                    'categories'
                );
            }
            $category = new Category();
            $category->name = $request->name;
            $category->menu_id = $request->menu_id;
            $category->image_url = $logoPath ?? '';
            $category->save();
            if ($request->has('subcategories') && is_array($request->subcategories)) {
                foreach ($request->subcategories as $subcat) {
                    $subcategory = new Category();
                    $subcategory->name = $subcat['nameSubcategory'];
                    $subcategory->parent_id = $category->category_id;
                    $subcategory->menu_id = $request->menu_id;
                    $subcategory->save();
                }
            }
            DB::commit();
            return response()->json([
                'message' => "Se creo la categoría exitosamente",
                'category' => $category
            ], Response::HTTP_OK);
        } catch (Exception $ex) {
            DB::rollBack();
            return response()->json(['error' => "Error al crear la categoría"], 500);
        }
    }

    public function updateCategory(Request $request)
    {
        try {
            if ($request->has('subcategories')) {
                $request->merge([
                    'subcategories' => json_decode($request->subcategories, true),
                ]);
            }
            $validatedData = Validator::make($request->all(), [
                'id' => 'required|integer|exists:categories,category_id',
                'name' => 'required|string|max:500',
                'img_file' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5048',
                'img_url' => 'nullable|string|max:600',
                'subcategories' => 'nullable|array',
                'subcategories.*.category_id' => 'nullable|integer|exists:categories,category_id',
                'subcategories.*.nameSubcategory' => 'required_with:subcategories|string|max:255',
            ]);
            if ($validatedData->fails()) {
                return response()->json(['error' => $validatedData->errors()->toArray()], 400);
            }
            DB::beginTransaction();
            $queryGetCategory = Category::where("category_id", $request->id)->first();
            if (!$queryGetCategory) {
                return response()->json(['error' => "No existe la categoría"], 500);
            }

            $imgPath = null;
            if ($request->hasFile('img_file')) {
                $imgPath = FileHelper::replace(
                    $request->file('img_file'),
                    $queryGetCategory->image_url,
                    'categories'
                );
            }

            if ((!$request->img_url && !$request->hasFile('img_file')) && $queryGetCategory->image_url) {
                FileHelper::delete($queryGetCategory->image_url);
            }
            $queryGetCategory->update([
                "name" => $request->name,
                "image_url" => ($imgPath) ? $imgPath : $request->img_url,
            ]);
            if ($request->has('subcategories') && is_array($request->subcategories)) {
                $idFilterNotDeleteSub = array_column($request->subcategories, 'category_id');
                $subcategoryQueryDelete = Category::where('parent_id', $queryGetCategory->category_id)
                ->whereNotIn('category_id', $idFilterNotDeleteSub)
                ->get();

                foreach ($subcategoryQueryDelete as $subcat) {
                    $subcat->delete();
                }

                foreach ($request->subcategories as $subcat) {
                    $idSubcategory = $subcat['category_id'];
                    $queryGetSubCategory = Category::where("category_id", $idSubcategory)->first();
                    if ($queryGetSubCategory) {
                        $queryGetSubCategory->name = $subcat['nameSubcategory'];
                        $queryGetSubCategory->save();
                    }else{
                        $subcategory = new Category();
                        $subcategory->name = $subcat['nameSubcategory'];
                        $subcategory->parent_id = $queryGetCategory->category_id;
                        $subcategory->menu_id = $queryGetCategory->menu_id;
                        $subcategory->save();
                    }
                }
            }
            DB::commit();
            return response()->json([
                'message' => "Se actualizo la categoría exitosamente",
            ], Response::HTTP_OK);
        } catch (Exception $ex) {
            DB::rollBack();
            return response()->json(['error' => "No se pudo actualizar la categoría"], 500);
        }
    }

    public function deleteCategory($id)
    {
        try {
            $deleteElement = Category::find($id);
            if (!$deleteElement) {
                return response()->json(['error' => "No existe la categoría"], 500);
            }
            FileHelper::delete($deleteElement->image_url);
            $deleteElement->delete();
            return response()->json([
                'message' => "Se eliminó la categoría exitosamente",
            ], Response::HTTP_OK);
        } catch (Exception $ex) {
            return response()->json(['error' => "Error al eliminar la categoría"], 500);
        }
    }
}