import './modal.js';
import flatpickr from "flatpickr";
import { Spanish } from "flatpickr/dist/l10n/es.js";

flatpickr(".only-date input", {
    dateFormat: "Y-m-d",
    enableTime: false,
    noCalendar: false,
    locale: Spanish
});

flatpickr(".only-time input", {
    dateFormat: "H:i",
    enableTime: true,
    noCalendar: true,
    locale: Spanish
});

/**
 * * Inicializa y mantiene el contador regresivo.
 * * @param {string} targetDateString La fecha y hora de destino en formato ISO (ej: "2026-12-25 10:00:00").
 * 
*/
function startCountdown(targetDateString) {

    // 1. Convertir la cadena de fecha/hora en un objeto Date de JavaScript
    const targetDate = new Date(targetDateString).getTime();

    // Elementos del DOM para mostrar el resultado
    const $days = document.getElementById('days');
    const $hours = document.getElementById('hours');
    const $minutes = document.getElementById('minutes');
    const $seconds = document.getElementById('seconds');
    const $message = document.getElementById('message');

    if (isNaN(targetDate)) {
        console.error("Error: La fecha de destino proporcionada no es válida.");
        document.getElementById('countdown-container').innerHTML = '<p class="text-red-500">Error al cargar la fecha del evento.</p>';
        return;
    }

    // 2. Función principal que se ejecutará cada segundo
    function updateCountdown() {
        const now = new Date().getTime();
        const distance = targetDate - now;

        // Si la cuenta regresiva ha terminado
        if (distance < 0) {
            clearInterval(interval);
            $days.textContent = '00';
            $hours.textContent = '00';
            $minutes.textContent = '00';
            $seconds.textContent = '00';
            $message.classList.remove('hidden'); // Mostrar mensaje de finalización
            return;
        }

        // Cálculos de tiempo
        const days = Math.floor(distance / (1000 * 60 * 60 * 24));
        const hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
        const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
        const seconds = Math.floor((distance % (1000 * 60)) / 1000);

        // Función auxiliar para agregar un cero inicial si el número es menor a 10
        const formatTime = (time) => time < 10 ? `0${time}` : time;

        // Actualizar el DOM
        $days.textContent = formatTime(days);
        $hours.textContent = formatTime(hours);
        $minutes.textContent = formatTime(minutes);
        $seconds.textContent = formatTime(seconds);
    }

    // 3. Ejecutar la función inmediatamente para evitar el parpadeo inicial con '0'
    updateCountdown();

    // 4. Configurar el intervalo para que se ejecute cada 1000ms (1 segundo)
    const interval = setInterval(updateCountdown, 1000);
}

window.startCountdown = startCountdown;

if (window.location.pathname === '/' || window.location.pathname === '/home') {
    import('./home.js').then(module => {
    });
    import('./calendar.js').then(module => {
    });
}