const getOS = () => {
    const ua = navigator.userAgent || navigator.vendor || window.opera;
    if (/android/i.test(ua)) return "android";
    if (/iPad|iPhone|iPod/.test(ua)) return "ios";
    return "other";
};

const isDesktop = () => window.innerWidth >= 1024;

const detectPreferredCalendar = () => {
    const os = getOS();
    if (os === "android") return "google";
    if (os === "ios") return "ios";
    if (isDesktop()) return "choose"; // mostrar opciones
    return "google"; // fallback
};

const calendarOutlook = (date, title, description, location) => {
    const start = new Date(date);

    // Fin del mismo día a las 11:59 PM
    const end = new Date(start);
    end.setHours(23, 59, 0, 0); // 11:59 PM

    const urlOutlook = `https://outlook.live.com/calendar/0/deeplink/compose?subject=${encodeURIComponent(
        title
    )}&body=${encodeURIComponent(description)}&location=${encodeURIComponent(
        location
    )}&startdt=${start.toISOString()}&enddt=${end.toISOString()}`;

    window.open(urlOutlook, "_blank");
};

const calendarGoogle = (date, title, description, location) => {
    const start = new Date(date);

    // Crear la fecha de fin a las 11:59 PM del mismo día
    const end = new Date(start);
    end.setHours(23, 59, 0, 0); // 11:59 PM

    const startStr = formatUTC(start);
    const endStr = formatUTC(end);

    const urlGoogle = `https://calendar.google.com/calendar/render?action=TEMPLATE&text=${encodeURIComponent(
        title
    )}&dates=${startStr}/${endStr}&details=${encodeURIComponent(
        description
    )}&location=${encodeURIComponent(location)}&ctz=America/Mexico_City`;

    window.open(urlGoogle, "_blank");
};

const calendarIOS = (date, title, description, location) => {
    const start = new Date(date);

    // Fin del día a las 11:59 PM
    const end = new Date(start);
    end.setHours(23, 59, 0, 0);

    const formatICS = (date) => {
        return date.toISOString().replace(/[-:]|\.\d{3}/g, "") + "Z";
    };

    const startStr = formatICS(start);
    const endStr = formatICS(end);
    const nowStr = formatICS(new Date());

    const escapeText = (text) =>
        text
            .replace(/(\r\n|\n|\r)/g, "\\n")
            .replace(/,/g, "\\,")
            .replace(/;/g, "\\;");

    const icsContent =
        "BEGIN:VCALENDAR\r\n" +
        "VERSION:2.0\r\n" +
        "PRODID:-//TuApp//EN\r\n" +
        "BEGIN:VEVENT\r\n" +
        `UID:${Date.now()}@tuapp.com\r\n` +
        `DTSTAMP:${nowStr}\r\n` +
        `DTSTART:${startStr}\r\n` +
        `DTEND:${endStr}\r\n` +
        `SUMMARY:${escapeText(title)}\r\n` +
        `DESCRIPTION:${escapeText(description)}\r\n` +
        `LOCATION:${escapeText(location)}\r\n` +
        "END:VEVENT\r\n" +
        "END:VCALENDAR";

    const blob = new Blob([icsContent], { type: "text/calendar" });
    const url = URL.createObjectURL(blob);

    const isIOS = /iPad|iPhone|iPod/.test(navigator.userAgent) && !window.MSStream;

    if (isIOS) {
        window.open(url, "_blank");
    } else {
        const link = document.createElement("a");
        link.href = url;
        link.download = "evento.ics";
        link.style.display = "none";
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
    }

    setTimeout(() => URL.revokeObjectURL(url), 10000);
};

const formatUTC = (d) => {
    return d.toISOString().replace(/[-:]|\.\d{3}/g, "").slice(0, 15) + "Z";
};

const btnCalendar = document.querySelectorAll(".btn-calendar").forEach((btn) => {
    btn.addEventListener("click", (e) => {
        e.preventDefault();
        const type = e.currentTarget.getAttribute("data-type-calendar");
        const titleMain = e.currentTarget.getAttribute("data-title");
        const descriptionMain = e.currentTarget.getAttribute("data-description");
        const locationMain = e.currentTarget.getAttribute("data-location");
        const startDate = e.currentTarget.getAttribute("data-start-date");
        const nowEvent = new Date(startDate);
        switch (type) {
            case 'ics':
                calendarIOS(nowEvent, titleMain, descriptionMain, locationMain);
                break;
            case 'outlook':
                alert("entra");
                calendarOutlook(nowEvent, titleMain, descriptionMain, locationMain);
                break;
            default:
                calendarGoogle(nowEvent, titleMain, descriptionMain, locationMain);
                break;
        }
    });
});