import { gsap } from "gsap";
import { ScrollTrigger } from "gsap/ScrollTrigger";
import PhotoSwipeLightbox from 'photoswipe/lightbox';
import 'photoswipe/style.css';

gsap.registerPlugin(ScrollTrigger);

const isLg = window.matchMedia("only screen and (min-width: 1024px)").matches;
const sliderModalTravels = new Swiper('.sliderModalTravels', {
  slidesPerView: 1,
  effect: 'slide',
  spaceBetween: 16,
});

/**
 * Anima el texto de un selector de clase línea por línea.
 * Dentro de cada línea, las letras aparecen ESCALONADAMENTE (stagger)
 * con opacidad de izquierda a derecha.
 *
 * @param {string} selector - El selector CSS (ej: '.mi-texto-animado').
 * @param {number} [duration=0.5] - Duración de la animación de CADA letra. (REDUCIDO)
 * @param {number} [charStagger=0.06] - Retraso entre CADA letra (el delay de izquierda a derecha). (REDUCIDO)
 * @param {number} [lineStagger=0.08] - Retraso entre la animación de CADA línea. (REDUCIDO)
 */
function animateTxtLeftToRigthOpacity(selector, duration = 0.3, charStagger = 0.06, lineStagger = 0.08) {
  // 1. Verificación de librerías.
  if (typeof gsap === 'undefined' || typeof SplitType === 'undefined') {
    console.error("GSAP o SplitType no están cargados. Asegúrate de incluirlos.");
    return;
  }

  const elements = document.querySelectorAll(selector);

  elements.forEach(element => {
    // 2. Dividir el texto por 'lines' Y 'chars' (letras)
    const splitText = new SplitType(element, {
      types: 'lines, chars',
      lineClass: 'line-animada'
    });

    // 3. Configurar el estado INICIAL de las letras: Opacidad 0
    gsap.set(splitText.chars, {
      opacity: 0,
    });

    // 4. Crear la Timeline con ScrollTrigger
    const tl = gsap.timeline({
      defaults: {
        ease: 'power2.out'
      },
      scrollTrigger: {
        trigger: element,
        start: 'top 80%',
        end: "top 80%",
        toggleActions: 'play none none none',
        markers: false
      }
    });

    // 5. Iterar sobre CADA LÍNEA para animar sus letras
    splitText.lines.forEach((line, index) => {
      const lineChars = line.querySelectorAll('.char');

      // 6. Aplicar la animación TO a las LETRAS DENTRO DE LA LÍNEA
      tl.to(lineChars, {
        opacity: 1, // Hace visible la letra
        duration: duration,
        // **CLAVE:** Usamos stagger para el delay entre letras (izquierda a derecha)
        stagger: charStagger,
      },
        // El tiempo de inicio de CADA LÍNEA (el escalonamiento línea por línea)
        index * lineStagger
      );
    });
  });
}

const animationFlowers = () => {
  const SECTION = document.getElementById('gift-section');
  const FLOWERS = document.querySelectorAll('.falling-flower-static');

  // Salir si faltan elementos
  if (!SECTION || FLOWERS.length === 0) return;

  // Calcular la altura del contenedor (SECTION) para usarlo como punto de inicio negativo (desde arriba).
  const SECTION_HEIGHT = SECTION.clientHeight;

  // 2. Configuración inicial de las flores estáticas
  FLOWERS.forEach(flower => {
    const targetFactor = parseFloat(flower.getAttribute('data-target-y')) || 1.0;
    const bottomPercentage = (1.0 - targetFactor) * 100;

    // Establecer posición absoluta para control manual
    flower.style.position = 'absolute';
    flower.style.bottom = `${bottomPercentage}%`;
  });

  /**
    * Crea y devuelve la línea de tiempo de animación para una sola flor.
    * @param {HTMLElement} flower El elemento DOM de la flor.
    */
  function createFlowerAnimation(flower) {

    const duration = 4; // Duración fija de la caída
    const delay = gsap.utils.random(0, 5); // Retraso aleatorio para escalonamiento
    const initialRotation = gsap.utils.random(360, 720);

    // 3a. Configuración inicial (Set)
    gsap.set(flower, {
      // Se usa el negativo de la altura de la sección para empezar desde ARRIBA del contenedor.
      y: -(SECTION_HEIGHT - flower.offsetHeight),
      rotation: initialRotation,
      opacity: 0,
    });

    // 3b. Crear Línea de Tiempo
    const tl = gsap.timeline({
      defaults: { ease: "linear" },
    });

    // Animación de opacidad a 1 al inicio
    tl.to(flower, {
      opacity: 1,
      duration: 0.5,
    }, delay);

    // Paso Único: Caída a la posición final (y: 0) y Rotación.
    tl.to(flower, {
      y: 0,
      rotation: 0,
      duration: duration,
    }, delay);

    return tl;
  }

  // 4. Crear línea de tiempo maestra
  const masterTimeline = gsap.timeline({ paused: true });

  FLOWERS.forEach(flower => {
    const flowerAnimation = createFlowerAnimation(flower);
    // Agrega todas las animaciones al mismo tiempo (posición 0)
    masterTimeline.add(flowerAnimation, 0);
  });

  // 5. ScrollTrigger para reproducir la animación al entrar a la vista
  ScrollTrigger.create({
    trigger: SECTION,
    start: "top bottom",
    end: "bottom top",
    markers: false,
    // Reproduce la animación una vez cuando la sección entra o re-entra.
    onEnter: () => masterTimeline.play(),
    onEnterBack: () => masterTimeline.play(),
  });
}

/**
 * Inicializa una animación de secuencia de frames en un canvas controlada por scroll,
 * seguida de la animación del contenido de la tarjeta, todo en una única línea de tiempo de GSAP.
 * @param {string} canvasSelector - Selector del elemento <canvas>.
 * @param {string} triggerSelector - Selector del elemento que actúa como punto de inicio y pin.
 * @param {string} contentSelector - Selector del contenedor del contenido de la card a animar.
 * @param {number} frameCount - El número total de frames (ej: 7).
 * @param {string} basePath - La ruta base a la carpeta de los frames (ej: '/images/assets-travel/frame-card/').
 * @param {string} fileExtension - La extensión de los archivos (ej: 'png').
 * @param {number} [pixelsPerFrame=400] - Distancia de scroll (en píxeles) para avanzar un frame.
 */
function initScrollFrameAnimation(canvasSelector, triggerSelector, contentSelector, frameCount, basePath, fileExtension, pixelsPerFrame = 400) {
  return new Promise((resolve) => { // Envuelve la lógica en una Promise

    // --- SETUP BÁSICO ---
    const canvas = document.querySelector(canvasSelector);
    const triggerElement = document.querySelector(triggerSelector);
    const contentElement = document.querySelector(contentSelector);

    if (!canvas || !triggerElement || !contentElement) {
      console.error("No se encontraron elementos Canvas, Trigger o Content. Revisa los selectores.");
      return resolve(); // Resuelve para no bloquear si faltan elementos
    }

    const context = canvas.getContext("2d");
    const frames = [];
    const animation = { frame: 0 };

    const framesScrollDistance = (frameCount - 1) * pixelsPerFrame;
    const cardAnimationScrollDistance = pixelsPerFrame * 3;
    const totalTriggerHeight = framesScrollDistance + cardAnimationScrollDistance;

    const getFramePath = (index) => {
      // Se usa i + 1 porque los archivos van de 'frame1' a 'frameN'
      return `${basePath}frame${index + 1}.${fileExtension}`;
    };

    const drawFrame = (index) => {
      const frameIndex = Math.round(index);
      const safeIndex = Math.min(Math.max(0, frameIndex), frameCount - 1);

      if (frames[safeIndex] && frames[safeIndex].complete) {
        context.clearRect(0, 0, canvas.width, canvas.height);
        context.drawImage(frames[safeIndex], 0, 0, canvas.width, canvas.height);
      }
    };

    // SETUP PRINCIPAL DE LA ANIMACIÓN CON TIMELINE
    const setupAnimation = () => {
      const tl = gsap.timeline({
        scrollTrigger: {
          trigger: triggerElement,
          scrub: 0.5,
          start: "top top",
          end: `+=${totalTriggerHeight}`,
          pin: true, // Esto fija el elemento y crea el espacio (placeholder)
        }
      });

      // 1. Animación de la Secuencia de Frames
      tl.to(animation, {
        frame: frameCount - 1,
        ease: "none",
        snap: "frame",
        duration: framesScrollDistance,
        onUpdate: () => drawFrame(animation.frame),
      }, 0);

      // 2. Animación del Contenido de la Card
      const startTime = framesScrollDistance * 0.9;
      gsap.set(contentElement, { transformOrigin: 'top center' });

      tl.to(contentElement, {
        opacity: 1,
        scaleY: 1,
        y: 0,
        duration: cardAnimationScrollDistance,
        ease: "power3.out"
      }, startTime);
    };

    const preloadImages = () => {
      let loadedCount = 0;

      for (let i = 0; i < frameCount; i++) {
        const img = new Image();

        img.onload = () => {
          loadedCount++;

          if (i === 0) {
            // Establecer dimensiones del canvas basadas en el primer frame
            canvas.width = img.width;
            canvas.height = img.height;
            drawFrame(0);
            canvas.style.objectFit = 'contain';
          }

          if (loadedCount === frameCount) {
            setupAnimation();
            resolve();
          }
        };

        img.onerror = () => {
          console.error('Error al cargar frame en initScrollFrameAnimation:', getFramePath(i));
          loadedCount++;
          if (loadedCount === frameCount) {
            setupAnimation();
            resolve(); // Resolver incluso en caso de error para no bloquear
          }
        };

        img.src = getFramePath(i);
        frames.push(img);
      }
    };

    preloadImages();
  });
}

/**
* Inicializa una animación de secuencia de frames en un canvas controlada por scroll,
* sincronizada con la transición de múltiples elementos de contenido (slides).
* La transición de los slides se centra alrededor de los puntos de 'pixelsPerTransition'.
* * @param {string} canvasSelector - Selector del elemento <canvas>.
* @param {string} triggerSelector - Selector del elemento que actúa como punto de inicio y pin.
* @param {string} slidesSelector - Selector de todos los contenedores de slides a animar.
* @param {number} frameCount - El número total de frames.
* @param {string} basePath - La ruta base a la carpeta de los frames.
* @param {number} pixelsPerTransition - Distancia de scroll reservada para cada sección de slide.
*/
function initTimeCanvasAnimation(canvasSelector, triggerSelector, slidesSelector, frameCount, basePath, pixelsPerTransition) {
  return new Promise((resolve) => { // Envuelve la lógica en una Promise
    // --- SELECTORES DE ELEMENTOS ---
    const canvas = document.querySelector(canvasSelector);
    const triggerElement = document.querySelector(triggerSelector);
    const slideElements = document.querySelectorAll(slidesSelector);

    // Validaciones iniciales
    if (!canvas || !triggerElement || slideElements.length === 0) {
      console.error("No se encontraron elementos Canvas, Trigger o Slides. Revisa los selectores.");
      return resolve();
    }
    if (typeof gsap === 'undefined' || typeof ScrollTrigger === 'undefined') {
      console.error("GSAP o ScrollTrigger no están cargados. Asegúrate de incluir las librerías.");
      return resolve();
    }

    const context = canvas.getContext("2d");
    const frames = [];
    const animation = { frame: 0 };

    const slideCount = slideElements.length;
    const totalTransitions = slideCount - 1;
    // La distancia total de scroll incluye todas las secciones de transición/espera
    // y una distancia extra al final para que el último frame/slide se mantenga.
    // Usamos el número de slides completos (slideCount) por pixelsPerTransition
    // El último slide tiene una "espera" de pixelsPerTransition.
    const totalScrollDistance = (slideCount) * pixelsPerTransition;

    const TRANSITION_SCROLL_LENGTH = pixelsPerTransition * 0.25;
    const TRANSITION_START_OFFSET = TRANSITION_SCROLL_LENGTH / 2;

    const getFramePath = (index) => {
      // Usamos el índice de frame + 1 para las URLs
      return `${basePath}frame${index + 1}.jpg`;
    };

    const drawFrame = (index) => {
      const frameIndex = Math.round(index);
      const safeIndex = Math.min(Math.max(0, frameIndex), frameCount - 1);

      if (frames[safeIndex] && frames[safeIndex].complete) {
        context.clearRect(0, 0, canvas.width, canvas.height);
        context.drawImage(frames[safeIndex], 0, 0, canvas.width, canvas.height);
      }
    };

    // SETUP PRINCIPAL DE LA ANIMACIÓN CON TIMELINE
    const setupAnimation = () => {
      // 1. Configuración inicial de los slides
      gsap.set(slideElements, {
        opacity: 0, y: 20, filter: "blur(5px)",
      });
      gsap.set(slideElements[0], { opacity: 1, y: 0, filter: "none" });

      // 2. Creación del Timeline principal
      const tl = gsap.timeline({
        scrollTrigger: {
          trigger: triggerElement,
          scrub: 0.5,
          start: "top top",
          end: `+=${totalScrollDistance}`,
          pin: true,
        }
      });

      // 3. Animación de la Secuencia de Frames
      tl.to(animation, {
        frame: frameCount - 1,
        ease: "none",
        duration: totalScrollDistance,
        onUpdate: () => drawFrame(animation.frame),
      }, 0);

      // 4. Animación de Transición de Slides
      for (let i = 0; i < totalTransitions; i++) {
        const currentSlide = slideElements[i];
        const nextSlide = slideElements[i + 1];
        const centerTime = (i + 1) * pixelsPerTransition;
        const transitionStartTime = centerTime - TRANSITION_START_OFFSET;

        tl.to(currentSlide, {
          opacity: 0, y: -20, filter: "blur(5px)", duration: TRANSITION_SCROLL_LENGTH, ease: "power2.in",
        }, transitionStartTime);

        tl.fromTo(nextSlide,
          { opacity: 0, y: 20, filter: "blur(5px)" },
          { opacity: 1, y: 0, filter: "none", duration: TRANSITION_SCROLL_LENGTH, ease: "power2.out", },
          transitionStartTime
        );
      }
    };

    // PRECARGA DE IMÁGENES
    const preloadImages = () => {
      let loadedCount = 0;

      for (let i = 0; i < frameCount; i++) {
        const img = new Image();
        img.crossOrigin = "anonymous";

        img.onload = () => {
          loadedCount++;

          if (i === 0) {
            // Establecer dimensiones del canvas basadas en el primer frame
            canvas.width = img.width;
            canvas.height = img.height;
            const container = canvas.parentElement;
            canvas.style.width = container.offsetWidth + 'px';
            canvas.style.height = container.offsetHeight + 'px';
            drawFrame(0);
          }

          if (loadedCount === frameCount) {
            setupAnimation();
            resolve();
          }
        };

        img.onerror = () => {
          console.error(`Error al cargar el frame en initTimeCanvasAnimation: ${getFramePath(i)}. Asegúrate de que las rutas sean correctas.`);
          loadedCount++;
          if (loadedCount === frameCount) {
            setupAnimation();
            resolve();
          }
        }

        img.src = getFramePath(i);
        frames.push(img);
      }
    };

    // Configuración para el redimensionamiento del canvas
    const resizeCanvas = () => {
      if (!canvas.width || !canvas.height) return;
      const container = canvas.parentElement;
      const width = container.offsetWidth;
      const height = container.offsetHeight;

      canvas.style.width = width + 'px';
      canvas.style.height = height + 'px';
      drawFrame(animation.frame);
    };
    window.addEventListener('resize', resizeCanvas);

    // Iniciar la precarga
    preloadImages();
  });
}

const galleryInit = () => {
  const galleryElement = document.getElementById('my-gallery');
  if (!galleryElement) return;

  const links = galleryElement.querySelectorAll('a');
  const promises = [];

  links.forEach(link => {
    const imgSrc = link.getAttribute('href');

    if (link.getAttribute('data-pswp-width') && link.getAttribute('data-pswp-height')) {
      return;
    }

    const imgPromise = new Promise((resolve) => {
      const img = new Image();

      img.onload = () => {
        link.setAttribute('data-pswp-width', img.naturalWidth);
        link.setAttribute('data-pswp-height', img.naturalHeight);
        resolve();
      };

      img.onerror = () => {
        console.error('Error al cargar la imagen para PhotoSwipe:', imgSrc);
        link.setAttribute('data-pswp-width', 1000);
        link.setAttribute('data-pswp-height', 800);
        resolve();
      };
      img.src = imgSrc;
    });

    promises.push(imgPromise);
  });

  Promise.all(promises)
    .then(() => {
      const lightbox = new PhotoSwipeLightbox({
        gallery: '#my-gallery',
        children: 'a',
        pswpModule: () => import('photoswipe')
      });

      lightbox.init();

    })
    .catch(error => {
      console.error('Error durante la inicialización de PhotoSwipe:', error);
    });
}

const topPage = () => {
  window.scrollTo(0, 0);
};

const btnScrollTop = () => {
  const btnScrol = document.getElementById('btn-scroll-top');
  if (!btnScrol) return;
  btnScrol.addEventListener('click', () => {
    topPage();
  });
};

/**
 * Copia texto al portapapeles mostrando retroalimentación visual.
 * 
 * @param {Object} options - Configuración con los IDs o elementos a usar.
 * @param {string|HTMLElement} options.textElement - ID o elemento del texto a copiar.
 * @param {string|HTMLElement} options.copyButton - ID o elemento del botón.
 * @param {string|HTMLElement} options.copyIcon - ID o elemento del icono de copiar.
 * @param {string|HTMLElement} options.successIcon - ID o elemento del icono de éxito.
 * @param {string|HTMLElement} options.successMessage - ID o elemento del mensaje de éxito.
 */
const clipboardCopy = ({
  textElement = '',
  copyButton = '',
  copyIcon = '',
  successIcon = '',
  successMessage = ''
} = {}) => {
  const getEl = (el) => typeof el === 'string' ? document.getElementById(el) : el;

  const textEl = getEl(textElement);
  const buttonEl = getEl(copyButton);
  const copyIconEl = getEl(copyIcon);
  const successIconEl = getEl(successIcon);
  const successMsgEl = getEl(successMessage);

  if (!textEl || !buttonEl) {
    console.warn('clipboardCopy: faltan elementos obligatorios');
    return;
  }

  const copyToClipboard = async () => {
    const textToCopy = textEl.textContent.trim();

    try {
      await navigator.clipboard.writeText(textToCopy);
      showSuccessFeedback();
    } catch (err) {
      console.error('Error al intentar copiar: ', err);
    }
  };

  const showSuccessFeedback = () => {
    copyIconEl?.classList.add('hidden');
    successIconEl?.classList.remove('hidden');

    successMsgEl?.classList.remove('opacity-0');
    successMsgEl?.classList.add('opacity-100');

    setTimeout(() => {
      copyIconEl?.classList.remove('hidden');
      successIconEl?.classList.add('hidden');

      successMsgEl?.classList.remove('opacity-100');
      successMsgEl?.classList.add('opacity-0');
    }, 2000);
  };

  buttonEl.addEventListener('click', copyToClipboard);
};

const animationPostCDMX = () => {
  const durationBase = 1.5;
  const rotationAmount = 15;

  const tl = gsap.timeline({
    repeat: -1,
    yoyo: true,
    ease: "back.inOut"
  });

  tl.to(".bolet1, .bolet2", {
    rotation: (i) => i % 2 === 0 ? rotationAmount : -rotationAmount,
    scale: 1.05,
    duration: durationBase,
    stagger: {
      each: 0.2,
      from: "start"
    }
  }, 0);

  gsap.to(".envolve", {
    rotation: 20,
    duration: durationBase,
    ease: "back.inOut",
    repeat: -1,
    yoyo: true
  });

  gsap.to(".bolet-green", {
    rotation: -20,
    duration: durationBase,
    ease: "back.inOut",
    repeat: -1,
    yoyo: true,
    delay: 2
  });

  gsap.to(".postal-scale", {
    scale: 1.1,
    y: 8,
    duration: durationBase,
    ease: "back.inOut",
    repeat: -1,
    yoyo: true,
    delay: 2.2
  });
};

const modalPosts = () => {
  const modal = document.querySelector('.modal-posts-slides');
  const cardTravels = document.querySelectorAll('.card-travel');
  const closeModalButtons = document.querySelectorAll('.close-modal-content');
  const indicadorMobile = document.getElementById('indicadorMobileSlide');

  const openModal = () => {
    const tlMenuOpen = gsap.timeline();
    tlMenuOpen.set("html", { "overflow-y": "hidden" });
    tlMenuOpen.set('#modal-slides-travel', { zIndex: 9999 });
    tlMenuOpen.to(modal, { y: "0%", duration: 0.6, ease: 'power3.out' });
  };

  const closeModal = () => {
    const tlMenu = gsap.timeline();
    tlMenu.to(modal, { y: "100%", duration: 0.6, ease: 'power3.out' });
    tlMenu.set("html", { "overflow-y": "unset" });
    tlMenu.set('#modal-slides-travel', { zIndex: -1 });
  };

  closeModalButtons.forEach(button => {
    button.addEventListener('click', closeModal);
  });

  cardTravels.forEach((card, index) => {
    card.addEventListener('click', () => {
      let slideNumber = card.getAttribute("data-slide");
      sliderModalTravels.slideTo(index, slideNumber);
      openModal();
      indicadorMobile.style.display = 'block';
      indicadorMobile.style.opacity = 0;
      gsap.to(indicadorMobile, { opacity: 1, duration: 0.5 });
      gsap.to(indicadorMobile, {
        opacity: 0,
        delay: 3, // se oculta después de 3 segundos
        duration: 0.5,
        onComplete: () => {
          indicadorMobile.style.display = 'none';
        }
      });
    });
  });
};

modalPosts();
clipboardCopy({
  textElement: 'text-to-copy',
  copyButton: 'copy-button',
  copyIcon: 'copy-icon',
  successIcon: 'success-icon',
  successMessage: 'success-message'
});

clipboardCopy({
  textElement: 'text-to-copy2',
  copyButton: 'copy-button2',
  copyIcon: 'copy-icon2',
  successIcon: 'success-icon2',
  successMessage: 'success-message2'
});

/**
 * Inicializa un efecto Parallax robusto en un elemento usando GSAP y ScrollTrigger.
 *
 * @param {string} selector - Selector CSS para el elemento (e.g., '#hero-img').
 * @param {number} startMovement - Define la posición Y inicial de la imagen (e.g., -100).
 * @param {number} endMovement - Define la posición Y final de la imagen (e.g., 100).
 * @param {string} triggerSelector - Selector del elemento que 'dispara' la animación (opcional, por defecto es el propio selector).
 */
function applyParallax(selector, startMovement = -200, endMovement = 200, triggerSelector = selector) {
  // Verificar si GSAP y ScrollTrigger están disponibles
  if (typeof gsap === 'undefined' || typeof ScrollTrigger === 'undefined') {
    console.error('GSAP o ScrollTrigger no están cargados. Asegúrate de incluir los scripts.');
    return;
  }

  gsap.fromTo(
    selector,
    { y: startMovement },
    {
      y: endMovement,
      ease: 'none',
      scrollTrigger: {
        trigger: triggerSelector,
        scrub: true,
        start: 'top bottom',
        end: 'bottom top',
        // markers: false,
      }
    }
  );
}

function animateBoatLeftWithScroll(selector, distance = 100, easeType = 'power1.inOut') {
  // Asegurarse de que GSAP y ScrollTrigger están cargados
  if (typeof gsap === 'undefined' || typeof ScrollTrigger === 'undefined') {
    console.error('GSAP o ScrollTrigger no están cargados. Incluye los scripts necesarios.');
    return;
  }

  gsap.to(
    selector,
    {
      x: -distance,
      ease: easeType,
      scrollTrigger: {
        trigger: selector,
        scrub: true,
        start: 'top bottom',
        end: 'bottom center',
        // markers: false
      }
    }
  );
}

galleryInit();
btnScrollTop();
topPage();
animationPostCDMX();

// 1. Iniciar ambas animaciones de Canvas (que ahora devuelven Promises)
const framesPathTime = (isLg) ? '/images/assets-travel/frame-time/' : '/images/assets-travel/frame-time-mobile/';

const frameAnimationPromise = initScrollFrameAnimation(
  '#animation-canvas',
  '#scroll-animation-container',
  '#card-info',
  25,
  '/images/assets-travel/frame-card/',
  'png',
  50
);

const timeAnimationPromise = initTimeCanvasAnimation(
  '#animation-canvas-time',
  '#time-canvas-weeding',
  '.slides-time',
  36,
  framesPathTime,
  1000
);

// 2. Esperar a que AMBAS promesas se resuelvan (precarga y pin creados).
Promise.all([frameAnimationPromise, timeAnimationPromise])
  .then(() => {
    // 3. Iniciar la animación de texto DESPUÉS de que los pines se hayan establecido.
    animateTxtLeftToRigthOpacity('.text-animated-opacity-split');

    // 4. Recalcular la posición de todos los ScrollTriggers después de que
    //    los elementos 'pin' hayan añadido su espacio al DOM.
    ScrollTrigger.refresh();

    // 5. Iniciar la animación de las flores de forma segura.
    animationFlowers();
    applyParallax('.img-parallax-flower', 1.5);
    animateBoatLeftWithScroll('.boat-animation-move');
  })
  .catch(error => {
    console.error("Una de las animaciones de Canvas falló en la carga. Iniciando las demás animaciones de todas formas.", error);
    // Ejecutar el resto aunque falle alguna carga
    animateTxtLeftToRigthOpacity('.text-animated-opacity-split');
    ScrollTrigger.refresh();
    animationFlowers();
    applyParallax('.img-parallax-flower', 1.5);
    animateBoatLeftWithScroll('.boat-animation-move');
  });